<?php
/**
 * WPG Agent Detector — Identifies the AI agent behind an MCP request.
 *
 * Detects the agent by examining MCP session data, User-Agent headers,
 * REST API metadata, and configurable agent profiles.
 *
 * @package aos-wp-governance
 * @since   1.0.0
 */

defined('ABSPATH') || exit;

class WPG_Agent_Detector
{

    /** @var WPG_Agent_Detector|null */
    private static ?WPG_Agent_Detector $instance = null;

    /** @var array Known agent signatures */
    private array $signatures;

    /** @var array|null Cached detection result for current request */
    private ?array $cached_detection = null;

    public static function instance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        $this->signatures = $this->get_known_signatures();
    }

    /**
     * Detect the current agent from the request context.
     *
     * @param \WP_REST_Request|null $request Optional REST request object.
     *
     * @return array{
     *     agent: string,
     *     agent_type: string,
     *     confidence: string,
     *     session_id: string,
     *     details: array
     * }
     */
    public function detect(?\WP_REST_Request $request = null): array
    {
        if (null !== $this->cached_detection) {
            return $this->cached_detection;
        }

        $result = [
            'agent' => 'unknown',
            'agent_type' => 'unknown',
            'confidence' => 'none',
            'session_id' => '',
            'details' => [],
        ];

        // 1. Check MCP session ID.
        $session_id = $this->get_mcp_session_id($request);
        if ($session_id) {
            $result['session_id'] = $session_id;
            $result['agent_type'] = 'mcp';
        }

        // 2. Check User-Agent header.
        $user_agent = $this->get_user_agent($request);
        if ($user_agent) {
            $result['details']['user_agent'] = $user_agent;

            foreach ($this->signatures as $sig) {
                if ($this->matches_signature($user_agent, $sig)) {
                    $result['agent'] = $sig['id'];
                    $result['confidence'] = 'high';
                    break;
                }
            }
        }

        // 3. Check route-based detection (for known MCP adapter routes).
        if ($request) {
            $route = $request->get_route();
            $result['details']['route'] = $route;

            if (str_contains($route, '/mcp/')) {
                $result['agent_type'] = 'mcp';
            }

            // Check for agent hint in request body.
            $body = $request->get_json_params();
            if (!empty($body['params']['arguments']['_agent'])) {
                $result['agent'] = sanitize_text_field($body['params']['arguments']['_agent']);
                $result['confidence'] = 'self-reported';
            }
        }

        // 4. If still unknown but has MCP session, mark as generic MCP agent.
        if ('unknown' === $result['agent'] && 'mcp' === $result['agent_type']) {
            $result['agent'] = 'mcp-agent';
            $result['confidence'] = 'low';
        }

        // 5. Check if this is a direct REST API call (not MCP).
        if ('unknown' === $result['agent_type'] && $request) {
            $result['agent_type'] = 'rest';
        }

        // 6. Apply custom agent profile overrides.
        $result = $this->apply_agent_profiles($result);

        /**
         * Filter the detected agent information.
         *
         * Allows third-party plugins or custom code to modify
         * or enhance agent detection.
         *
         * @since 1.0.0
         *
         * @param array                $result  The detection result.
         * @param \WP_REST_Request|null $request The REST request, if available.
         */
        $result = apply_filters('wpg_detected_agent', $result, $request);

        $this->cached_detection = $result;

        return $result;
    }

    /**
     * Clear the cached detection (for new requests).
     */
    public function reset(): void
    {
        $this->cached_detection = null;
    }

    /**
     * Get the MCP Session ID from the request headers.
     *
     * @param \WP_REST_Request|null $request The REST request.
     *
     * @return string The session ID, or empty string.
     */
    private function get_mcp_session_id(?\WP_REST_Request $request = null): string
    {
        // Check REST request headers first.
        if ($request) {
            $header = $request->get_header('mcp-session-id')
                ?? $request->get_header('mcp_session_id');
            if ($header) {
                return sanitize_text_field($header);
            }
        }

        // Fall back to $_SERVER.
        $server_keys = ['HTTP_MCP_SESSION_ID', 'HTTP_MCP-SESSION-ID'];
        foreach ($server_keys as $key) {
            if (!empty($_SERVER[$key])) {
                return sanitize_text_field(wp_unslash($_SERVER[$key]));
            }
        }

        return '';
    }

    /**
     * Get the User-Agent header from the request.
     *
     * @param \WP_REST_Request|null $request The REST request.
     *
     * @return string The User-Agent, or empty string.
     */
    private function get_user_agent(?\WP_REST_Request $request = null): string
    {
        if ($request) {
            $ua = $request->get_header('user-agent');
            if ($ua) {
                return sanitize_text_field($ua);
            }
        }

        return sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'] ?? ''));
    }

    /**
     * Check if a User-Agent string matches a known agent signature.
     *
     * @param string $user_agent The User-Agent string.
     * @param array  $signature  The agent signature to match against.
     *
     * @return bool Whether the User-Agent matches.
     */
    private function matches_signature(string $user_agent, array $signature): bool
    {
        $ua_lower = strtolower($user_agent);

        foreach ((array)$signature['patterns'] as $pattern) {
            if (str_contains($ua_lower, strtolower($pattern))) {
                return true;
            }
        }

        return false;
    }

    /**
     * Apply custom agent profiles (admin-configured overrides).
     *
     * @param array $result The detection result.
     *
     * @return array Modified detection result.
     */
    private function apply_agent_profiles(array $result): array
    {
        $profiles = get_option('wpg_agent_profiles', []);

        if (empty($profiles) || !is_array($profiles)) {
            return $result;
        }

        foreach ($profiles as $profile) {
            if (!empty($profile['match_agent']) && $profile['match_agent'] === $result['agent']) {
                // Override with profile settings.
                $result['agent'] = $profile['alias'] ?? $result['agent'];
                $result['details']['profile_applied'] = $profile['name'] ?? '';
                break;
            }
        }

        return $result;
    }

    /**
     * Get known agent signature definitions.
     *
     * @return array Array of signature definitions.
     */
    private function get_known_signatures(): array
    {
        $signatures = [
            [
                'id' => 'claude-desktop',
                'name' => 'Claude Desktop',
                'vendor' => 'Anthropic',
                'patterns' => ['claude-desktop', 'anthropic', 'claude'],
            ],
            [
                'id' => 'cursor',
                'name' => 'Cursor IDE',
                'vendor' => 'Cursor',
                'patterns' => ['cursor', 'cursor-ide'],
            ],
            [
                'id' => 'windsurf',
                'name' => 'Windsurf',
                'vendor' => 'Codeium',
                'patterns' => ['windsurf', 'codeium'],
            ],
            [
                'id' => 'vscode-copilot',
                'name' => 'VS Code Copilot',
                'vendor' => 'Microsoft',
                'patterns' => ['vscode', 'visual-studio-code', 'copilot'],
            ],
            [
                'id' => 'chatgpt',
                'name' => 'ChatGPT',
                'vendor' => 'OpenAI',
                'patterns' => ['chatgpt', 'openai'],
            ],
            [
                'id' => 'gemini',
                'name' => 'Gemini',
                'vendor' => 'Google',
                'patterns' => ['gemini', 'google-ai'],
            ],
            [
                'id' => 'wp-cli',
                'name' => 'WP-CLI',
                'vendor' => 'WordPress',
                'patterns' => ['wp-cli'],
            ],
        ];

        /**
         * Filter the known agent signatures.
         *
         * Allows adding custom agent detection patterns.
         *
         * @since 1.0.0
         *
         * @param array $signatures The agent signatures.
         */
        return apply_filters('wpg_agent_signatures', $signatures);
    }
}
