<?php
/**
 * WPG Audit Log — Immutable, append-only audit trail.
 *
 * Every policy evaluation — whether allowed or denied — is recorded with full
 * context. Entries cannot be modified or deleted through the plugin interface.
 *
 * @package aos-wp-governance
 * @since   1.0.0
 */

defined('ABSPATH') || exit;

class WPG_Audit_Log
{

    /** @var WPG_Audit_Log|null */
    private static ?WPG_Audit_Log $instance = null;

    /** @var string The database table name (without prefix) */
    private const TABLE_NAME = 'wpg_audit_log';

    /** @var string Full table name with prefix, set in constructor */
    private string $table;

    public static function instance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        global $wpdb;
        $this->table = $wpdb->prefix . self::TABLE_NAME;
    }

    /**
     * Create the audit log database table.
     *
     * Uses dbDelta for safe creation and upgrades.
     */
    public function create_table(): void
    {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$this->table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            timestamp DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            decision VARCHAR(10) NOT NULL DEFAULT 'allow',
            ability VARCHAR(255) NOT NULL DEFAULT '',
            agent VARCHAR(255) NOT NULL DEFAULT 'unknown',
            agent_type VARCHAR(50) NOT NULL DEFAULT 'unknown',
            user_id BIGINT UNSIGNED NOT NULL DEFAULT 0,
            session_id VARCHAR(255) NOT NULL DEFAULT '',
            ip_address VARCHAR(45) NOT NULL DEFAULT '',
            policy_id VARCHAR(255) NOT NULL DEFAULT '',
            policy_name VARCHAR(255) NOT NULL DEFAULT '',
            reason TEXT NOT NULL,
            args_json LONGTEXT NOT NULL,
            matched_json LONGTEXT NOT NULL,
            elapsed_ms DECIMAL(10,3) NOT NULL DEFAULT 0,
            checksum VARCHAR(64) NOT NULL DEFAULT '',
            PRIMARY KEY (id),
            KEY idx_timestamp (timestamp),
            KEY idx_decision (decision),
            KEY idx_ability (ability),
            KEY idx_agent (agent),
            KEY idx_user_id (user_id),
            KEY idx_session_id (session_id),
            KEY idx_policy_id (policy_id)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    /**
     * Record a policy evaluation result to the audit log.
     *
     * This is an append-only operation. Entries cannot be modified.
     *
     * @param WPG_Policy_Result $result The evaluation result.
     *
     * @return int|false The inserted row ID, or false on failure.
     */
    public function record(WPG_Policy_Result $result): int|false
    {
        global $wpdb;

        $args_json = wp_json_encode($result->context['args'] ?? []);
        $matched_json = wp_json_encode($result->matched);

        // Generate a tamper-evident checksum.
        $checksum = $this->compute_checksum($result);

        $inserted = $wpdb->insert(
            $this->table,
        [
            'timestamp' => $result->context['timestamp'] ?? current_time('mysql', true),
            'decision' => $result->decision,
            'ability' => $result->context['ability'] ?? '',
            'agent' => $result->context['agent'] ?? 'unknown',
            'agent_type' => $result->context['agent_type'] ?? 'unknown',
            'user_id' => $result->context['user_id'] ?? 0,
            'session_id' => $result->context['session_id'] ?? '',
            'ip_address' => $result->context['ip'] ?? '',
            'policy_id' => $result->policy_id,
            'policy_name' => $result->policy_name,
            'reason' => $result->reason,
            'args_json' => $args_json,
            'matched_json' => $matched_json,
            'elapsed_ms' => round($result->elapsed_ms, 3),
            'checksum' => $checksum,
        ],
        [
            '%s', '%s', '%s', '%s', '%s', '%d', '%s',
            '%s', '%s', '%s', '%s', '%s', '%s', '%f', '%s',
        ]
        );

        if (false === $inserted) {
            return false;
        }

        $row_id = (int)$wpdb->insert_id;

        /**
         * Fires after an audit log entry is recorded.
         *
         * @since 1.0.0
         *
         * @param int               $row_id  The audit log entry ID.
         * @param WPG_Policy_Result $result  The evaluation result.
         */
        do_action('wpg_audit_logged', $row_id, $result);

        return $row_id;
    }

    /**
     * Query the audit log with filters.
     *
     * @param array $filters {
     *     Optional. Query filters.
     *
     *     @type string $decision   Filter by decision ('allow', 'deny').
     *     @type string $ability    Filter by ability name (supports wildcards).
     *     @type string $agent      Filter by agent identifier.
     *     @type int    $user_id    Filter by WordPress user ID.
     *     @type string $session_id Filter by MCP session ID.
     *     @type string $policy_id  Filter by policy ID.
     *     @type string $from       Start date (Y-m-d H:i:s).
     *     @type string $to         End date (Y-m-d H:i:s).
     *     @type int    $per_page   Results per page (default 50, max 200).
     *     @type int    $page       Page number (default 1).
     *     @type string $orderby    Column to order by (default 'timestamp').
     *     @type string $order      Order direction: 'ASC' or 'DESC' (default 'DESC').
     * }
     *
     * @return array{entries: array, total: int, page: int, per_page: int, pages: int}
     */
    public function query(array $filters = []): array
    {
        global $wpdb;

        $where = [];
        $params = [];

        // Decision filter.
        if (!empty($filters['decision'])) {
            $where[] = 'decision = %s';
            $params[] = $filters['decision'];
        }

        // Ability filter (exact or LIKE for wildcards).
        if (!empty($filters['ability'])) {
            if (str_contains($filters['ability'], '*')) {
                $where[] = 'ability LIKE %s';
                $params[] = str_replace('*', '%', $filters['ability']);
            }
            else {
                $where[] = 'ability = %s';
                $params[] = $filters['ability'];
            }
        }

        // Agent filter.
        if (!empty($filters['agent'])) {
            $where[] = 'agent = %s';
            $params[] = $filters['agent'];
        }

        // User ID filter.
        if (isset($filters['user_id'])) {
            $where[] = 'user_id = %d';
            $params[] = (int)$filters['user_id'];
        }

        // Session ID filter.
        if (!empty($filters['session_id'])) {
            $where[] = 'session_id = %s';
            $params[] = $filters['session_id'];
        }

        // Policy ID filter.
        if (!empty($filters['policy_id'])) {
            $where[] = 'policy_id = %s';
            $params[] = $filters['policy_id'];
        }

        // Date range.
        if (!empty($filters['from'])) {
            $where[] = 'timestamp >= %s';
            $params[] = $filters['from'];
        }

        if (!empty($filters['to'])) {
            $where[] = 'timestamp <= %s';
            $params[] = $filters['to'];
        }

        // Build WHERE clause.
        $where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

        // Pagination.
        $per_page = min(max((int)($filters['per_page'] ?? 50), 1), 200);
        $page = max((int)($filters['page'] ?? 1), 1);
        $offset = ($page - 1) * $per_page;

        // Ordering.
        $allowed_orderby = ['timestamp', 'decision', 'ability', 'agent', 'elapsed_ms'];
        $orderby = in_array($filters['orderby'] ?? 'timestamp', $allowed_orderby, true)
            ? $filters['orderby'] ?? 'timestamp'
            : 'timestamp';

        $order = strtoupper($filters['order'] ?? 'DESC') === 'ASC' ? 'ASC' : 'DESC';

        // Count total.
        $count_sql = "SELECT COUNT(*) FROM {$this->table} {$where_sql}";
        if (!empty($params)) {
            $count_sql = $wpdb->prepare($count_sql, ...$params);
        }
        $total = (int)$wpdb->get_var($count_sql);

        // Fetch entries.
        $query_sql = "SELECT * FROM {$this->table} {$where_sql} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";
        $query_params = array_merge($params, [$per_page, $offset]);
        $entries = $wpdb->get_results(
            $wpdb->prepare($query_sql, ...$query_params),
            ARRAY_A
        );

        // Parse JSON fields.
        $entries = array_map(function ($row) {
            $row['args'] = json_decode($row['args_json'] ?? '[]', true);
            $row['matched'] = json_decode($row['matched_json'] ?? '[]', true);
            unset($row['args_json'], $row['matched_json']);
            return $row;
        }, $entries ?: []);

        return [
            'entries' => $entries,
            'total' => $total,
            'page' => $page,
            'per_page' => $per_page,
            'pages' => (int)ceil($total / $per_page),
        ];
    }

    /**
     * Get summary statistics for the audit log.
     *
     * @param string $period 'day', 'week', 'month', or 'all'.
     *
     * @return array{total: int, allowed: int, denied: int, top_agents: array, top_abilities: array}
     */
    public function get_stats(string $period = 'day'): array
    {
        global $wpdb;

        $from = match ($period) {
                'day' => gmdate('Y-m-d 00:00:00', strtotime('-1 day')),
                'week' => gmdate('Y-m-d 00:00:00', strtotime('-7 days')),
                'month' => gmdate('Y-m-d 00:00:00', strtotime('-30 days')),
                default => '1970-01-01 00:00:00',
            };

        $where = $wpdb->prepare('WHERE timestamp >= %s', $from);

        $total = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$this->table} {$where}");

        $allowed = (int)$wpdb->get_var(
            $wpdb->prepare("SELECT COUNT(*) FROM {$this->table} WHERE timestamp >= %s AND decision = 'allow'", $from)
        );

        $denied = (int)$wpdb->get_var(
            $wpdb->prepare("SELECT COUNT(*) FROM {$this->table} WHERE timestamp >= %s AND decision = 'deny'", $from)
        );

        $top_agents = $wpdb->get_results(
            $wpdb->prepare(
            "SELECT agent, COUNT(*) as count FROM {$this->table} WHERE timestamp >= %s GROUP BY agent ORDER BY count DESC LIMIT 10",
            $from
        ),
            ARRAY_A
        ) ?: [];

        $top_abilities = $wpdb->get_results(
            $wpdb->prepare(
            "SELECT ability, COUNT(*) as count, SUM(decision = 'deny') as denied FROM {$this->table} WHERE timestamp >= %s GROUP BY ability ORDER BY count DESC LIMIT 10",
            $from
        ),
            ARRAY_A
        ) ?: [];

        return [
            'period' => $period,
            'total' => $total,
            'allowed' => $allowed,
            'denied' => $denied,
            'deny_rate' => $total > 0 ? round(($denied / $total) * 100, 1) : 0,
            'top_agents' => $top_agents,
            'top_abilities' => $top_abilities,
        ];
    }

    /**
     * Verify the integrity of an audit log entry.
     *
     * @param int $entry_id The entry ID.
     *
     * @return bool Whether the checksum is valid.
     */
    public function verify_integrity(int $entry_id): bool
    {
        global $wpdb;

        $row = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$this->table} WHERE id = %d", $entry_id),
            ARRAY_A
        );

        if (!$row) {
            return false;
        }

        $expected = $row['checksum'];
        $row_data = $row['decision'] . $row['ability'] . $row['agent'] . $row['user_id']
            . $row['policy_id'] . $row['reason'] . $row['args_json'] . $row['timestamp'];

        $computed = hash('sha256', $row_data . wp_salt('auth'));

        return hash_equals($expected, $computed);
    }

    /**
     * Compute a tamper-evident checksum for an audit entry.
     *
     * @param WPG_Policy_Result $result The evaluation result.
     *
     * @return string SHA-256 checksum.
     */
    private function compute_checksum(WPG_Policy_Result $result): string
    {
        $data = $result->decision
            . ($result->context['ability'] ?? '')
            . ($result->context['agent'] ?? '')
            . ($result->context['user_id'] ?? 0)
            . $result->policy_id
            . $result->reason
            . wp_json_encode($result->context['args'] ?? [])
            . ($result->context['timestamp'] ?? '');

        return hash('sha256', $data . wp_salt('auth'));
    }

    /**
     * Get the table name.
     *
     * @return string Full table name.
     */
    public function get_table_name(): string
    {
        return $this->table;
    }
}
