<?php
/**
 * WPG Gate — The Constitutional Policy Gate.
 *
 * This is the enforcement point. It hooks into WordPress's ability execution
 * pipeline and REST API to intercept, evaluate, and enforce governance policies
 * before any AI agent action touches the site.
 *
 * The Gate is the single point where the line "The AI doesn't decide. The gate
 * enforces." becomes real code.
 *
 * @package aos-wp-governance
 * @since   1.0.0
 */

defined('ABSPATH') || exit;

class WPG_Gate
{

    /** @var WPG_Gate|null */
    private static ?WPG_Gate $instance = null;

    /** @var bool Whether the gate is currently active (prevents recursion) */
    private bool $active = true;

    /** @var bool Whether we're inside our own ability execution (prevents self-gating) */
    private bool $self_executing = false;

    public static function instance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        // Hook 1: Intercept ability execution via the wp_register_ability_args filter.
        // This wraps the ability's execution callback with our governance gate.
        add_filter('wp_register_ability_args', [$this, 'wrap_ability_callback'], 5, 2);

        // Hook 2: Intercept REST API calls to the MCP adapter routes.
        // This is the belt to the above suspenders — catches anything that goes through REST.
        add_filter('rest_pre_dispatch', [$this, 'intercept_rest_request'], 5, 3);

        // Hook 3: Listen to wp_before_execute_ability for logging even if we can't deny.
        add_action('wp_before_execute_ability', [$this, 'on_before_execute'], 10, 2);

        // Reset agent detection on each new request.
        add_action('rest_api_init', function () {
            WPG_Agent_Detector::instance()->reset();
        });
    }

    /**
     * Wrap an ability's execution callback with the governance gate.
     *
     * This is the PRIMARY enforcement mechanism. By intercepting at the
     * wp_register_ability_args filter, we wrap the ability's callback
     * so our policy evaluation runs BEFORE the original code executes.
     *
     * @param array  $args         The ability registration arguments.
     * @param string $ability_name The ability name (e.g., 'core/delete-posts').
     *
     * @return array Modified arguments with wrapped callback.
     */
    public function wrap_ability_callback(array $args, string $ability_name): array
    {
        // Don't gate our own abilities (prevent recursion).
        if (str_starts_with($ability_name, 'wpg/')) {
            return $args;
        }

        // Store the original callback.
        $original_callback = $args['callback'] ?? null;

        if (!is_callable($original_callback)) {
            return $args;
        }

        // Replace it with our gated version.
        $args['callback'] = function ($input) use ($original_callback, $ability_name) {
            return $this->gated_execute($ability_name, $input, $original_callback);
        };

        return $args;
    }

    /**
     * Execute an ability through the governance gate.
     *
     * @param string   $ability_name     The ability being executed.
     * @param mixed    $input            The input arguments.
     * @param callable $original_callback The original ability callback.
     *
     * @return mixed The ability result, or a WP_Error if denied.
     */
    private function gated_execute(string $ability_name, mixed $input, callable $original_callback): mixed
    {
        // Skip if gate is disabled or we're self-executing.
        if (!$this->active || $this->self_executing) {
            return call_user_func($original_callback, $input);
        }

        // Skip if governance is globally paused.
        if ($this->is_paused()) {
            return call_user_func($original_callback, $input);
        }

        // Detect the agent.
        $agent_info = WPG_Agent_Detector::instance()->detect();

        // Build the evaluation context.
        $context = [
            'ability' => $ability_name,
            'args' => is_array($input) ? $input : ['value' => $input],
            'agent' => $agent_info['agent'],
            'agent_type' => $agent_info['agent_type'],
            'user_id' => get_current_user_id(),
            'ip' => $this->get_client_ip(),
            'session_id' => $agent_info['session_id'],
            'timestamp' => current_time('mysql', true),
        ];

        // Evaluate against policies.
        $result = WPG_Policy_Engine::instance()->evaluate($context);

        // Record to audit log.
        WPG_Audit_Log::instance()->record($result);

        if ($result->is_denied()) {
            /**
             * Fires when an AI action is denied by a governance policy.
             *
             * @since 1.0.0
             *
             * @param WPG_Policy_Result $result The denial result.
             * @param string            $ability_name The ability that was denied.
             */
            do_action('wpg_action_denied', $result, $ability_name);

            // Fire alerts for deny events.
            $this->fire_alerts($result, $ability_name);

            return $result->to_wp_error();
        }

        /**
         * Fires when an AI action is approved by governance.
         *
         * @since 1.0.0
         *
         * @param WPG_Policy_Result $result The approval result.
         * @param string            $ability_name The ability that was approved.
         */
        do_action('wpg_action_approved', $result, $ability_name);

        // Execute the original callback.
        return call_user_func($original_callback, $input);
    }

    /**
     * Intercept REST API requests to the MCP adapter.
     *
     * This catches MCP requests that come through the REST API, providing
     * a secondary enforcement point in case the ability callback wrapping
     * doesn't catch something.
     *
     * @param mixed            $result  Pre-dispatch result (null to continue).
     * @param \WP_REST_Server  $server  The REST server instance.
     * @param \WP_REST_Request $request The REST request.
     *
     * @return mixed|WP_Error The original result or a WP_Error if denied.
     */
    public function intercept_rest_request(mixed $result, \WP_REST_Server $server, \WP_REST_Request $request): mixed
    {
        // If already handled, pass through.
        if (null !== $result) {
            return $result;
        }

        // Only intercept MCP adapter routes that execute abilities.
        $route = $request->get_route();
        if (!$this->is_mcp_execute_route($route)) {
            return $result;
        }

        // Skip if gate is disabled or paused.
        if (!$this->active || $this->is_paused()) {
            return $result;
        }

        // Extract ability info from the request body.
        $body = $request->get_json_params();
        $ability_name = $this->extract_ability_from_request($body);

        if (empty($ability_name)) {
            return $result; // Can't determine ability — let it through.
        }

        // Detect the agent.
        $agent_info = WPG_Agent_Detector::instance()->detect($request);

        // Build context.
        $context = [
            'ability' => $ability_name,
            'args' => $body['params']['arguments'] ?? [],
            'agent' => $agent_info['agent'],
            'agent_type' => 'mcp',
            'user_id' => get_current_user_id(),
            'ip' => $this->get_client_ip(),
            'session_id' => $agent_info['session_id'],
            'timestamp' => current_time('mysql', true),
        ];

        // Evaluate.
        $eval_result = WPG_Policy_Engine::instance()->evaluate($context);

        // Log the evaluation.
        WPG_Audit_Log::instance()->record($eval_result);

        if ($eval_result->is_denied()) {
            do_action('wpg_action_denied', $eval_result, $ability_name);
            $this->fire_alerts($eval_result, $ability_name);

            return $eval_result->to_wp_error();
        }

        // Allow through — the ability callback wrapper will also run,
        // but it will see the same result from the cached evaluation.
        return $result;
    }

    /**
     * Logging hook for wp_before_execute_ability action.
     *
     * This action fires in WordPress core but doesn't support returning
     * a value to deny the action. We use it for supplementary logging.
     *
     * @param string $ability_name The ability name.
     * @param mixed  $input        The input data.
     */
    public function on_before_execute(string $ability_name, mixed $input): void
    {
        // This is primarily for logging and monitoring.
        // Actual enforcement happens via wrap_ability_callback.

        /**
         * Fires when AOS detects an ability about to execute.
         *
         * @since 1.0.0
         *
         * @param string $ability_name The ability being executed.
         * @param mixed  $input        The input data.
         */
        do_action('wpg_ability_executing', $ability_name, $input);
    }

    /**
     * Check if a route is an MCP adapter ability execution route.
     *
     * @param string $route The REST API route.
     *
     * @return bool Whether this is an MCP execution route.
     */
    private function is_mcp_execute_route(string $route): bool
    {
        // Match MCP adapter execution patterns.
        return (bool)preg_match(
            '#^/mcp/[^/]+(/|$)#',
            $route
        );
    }

    /**
     * Extract the ability name from an MCP request body.
     *
     * @param array $body The JSON request body.
     *
     * @return string The ability name, or empty string.
     */
    private function extract_ability_from_request(array $body): string
    {
        // Direct ability name in params.
        if (!empty($body['params']['arguments']['ability_name'])) {
            return sanitize_text_field($body['params']['arguments']['ability_name']);
        }

        // MCP tool name might contain ability info.
        if (!empty($body['params']['name'])) {
            $tool_name = $body['params']['name'];

            // mcp-adapter-execute-ability -> extract from arguments.
            if (str_contains($tool_name, 'execute-ability')) {
                return sanitize_text_field(
                    $body['params']['arguments']['ability_name'] ?? ''
                );
            }

            return sanitize_text_field($tool_name);
        }

        return '';
    }

    /**
     * Fire alerts for denied actions.
     *
     * @param WPG_Policy_Result $result       The denial result.
     * @param string            $ability_name The denied ability.
     */
    private function fire_alerts(WPG_Policy_Result $result, string $ability_name): void
    {
        $alert_settings = get_option('wpg_alert_settings', []);

        // Email alert.
        if (!empty($alert_settings['email_enabled'])) {
            $to = $alert_settings['email_to'] ?? get_option('admin_email');

            wp_mail(
                $to,
                sprintf(
                /* translators: 1: Site name, 2: Ability name */
                __('[WPG Alert] Action denied on %1$s: %2$s', 'aos-wp-governance'),
                get_bloginfo('name'),
                $ability_name
            ),
                sprintf(
                "Policy: %s\nReason: %s\nAgent: %s\nAbility: %s\nTimestamp: %s",
                $result->policy_name,
                $result->reason,
                $result->context['agent'] ?? 'unknown',
                $ability_name,
                $result->context['timestamp'] ?? gmdate('Y-m-d H:i:s')
            )
            );
        }

        // Webhook alert.
        if (!empty($alert_settings['webhook_enabled']) && !empty($alert_settings['webhook_url'])) {
            wp_remote_post($alert_settings['webhook_url'], [
                'timeout' => 5,
                'body' => wp_json_encode([
                    'event' => 'wpg_action_denied',
                    'site' => get_bloginfo('url'),
                    'ability' => $ability_name,
                    'policy' => $result->policy_name,
                    'reason' => $result->reason,
                    'agent' => $result->context['agent'] ?? 'unknown',
                    'time' => $result->context['timestamp'] ?? gmdate('Y-m-d H:i:s'),
                ]),
                'headers' => ['Content-Type' => 'application/json'],
            ]);
        }

        /**
         * Fires when a governance alert is triggered.
         *
         * @since 1.0.0
         *
         * @param WPG_Policy_Result $result The denial result.
         * @param string $ability_name The denied ability.
         * @param array $alert_settings The configured alert settings.
         */
        do_action('wpg_alert_fired', $result, $ability_name, $alert_settings);
    }

    /**
     * Temporarily disable the gate (for internal operations).
     */
    public function pause(): void
    {
        $this->self_executing = true;
    }

    /**
     * Re-enable the gate after internal operations.
     */
    public function resume(): void
    {
        $this->self_executing = false;
    }

    /**
     * Check if governance is globally paused (e.g., during import).
     *
     * @return bool
     */
    private function is_paused(): bool
    {
        return (bool)get_transient('wpg_governance_paused');
    }

    /**
     * Get client IP address.
     *
     * @return string
     */
    private function get_client_ip(): string
    {
        foreach (['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'] as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = explode(',', sanitize_text_field(wp_unslash($_SERVER[$key])));
                return trim($ip[0]);
            }
        }
        return '127.0.0.1';
    }
}
