<?php
/**
 * WPG Governance Engine - Deterministic Constitutional Policy Evaluation.
 * @package aos-wp-governance
 * @since   1.0.0
 */
defined( 'ABSPATH' ) || exit;

class WPG_Policy_Result {
    public readonly string $decision;
    public readonly string $policy_id;
    public readonly string $policy_name;
    public readonly string $reason;
    public readonly float  $elapsed_ms;
    public readonly array  $context;
    public readonly array  $matched;

    public function __construct(
        string $decision, string $policy_id, string $policy_name,
        string $reason, float $elapsed_ms, array $context = [], array $matched = []
    ) {
        $this->decision    = $decision;
        $this->policy_id   = $policy_id;
        $this->policy_name = $policy_name;
        $this->reason      = $reason;
        $this->elapsed_ms  = $elapsed_ms;
        $this->context     = $context;
        $this->matched     = $matched;
    }

    public function is_denied(): bool { return 'deny' === $this->decision; }
    public function is_allowed(): bool { return 'allow' === $this->decision; }

    public function to_wp_error(): \WP_Error {
        return new \WP_Error( 'wpg_policy_denied', $this->reason, [
            'status' => 403, 'policy_id' => $this->policy_id,
            'policy_name' => $this->policy_name, 'elapsed_ms' => $this->elapsed_ms,
        ]);
    }

    public function to_array(): array {
        return [
            'decision'    => $this->decision,
            'policy_id'   => $this->policy_id,
            'policy_name' => $this->policy_name,
            'reason'      => $this->reason,
            'elapsed_ms'  => $this->elapsed_ms,
        ];
    }
}
class WPG_Policy_Engine {
    private static ?WPG_Policy_Engine $instance = null;
    private array $policies = [];
    private bool $loaded = false;

    public static function instance(): self {
        if ( null === self::$instance ) { self::$instance = new self(); }
        return self::$instance;
    }
    private function __construct() {}

    public function evaluate( array $context ): WPG_Policy_Result {
        $start = hrtime( true );
        $this->ensure_loaded();
        $ability = $context['ability'] ?? '';

        foreach ( $this->policies as $policy ) {
            if ( empty( $policy['active'] ) ) { continue; }
            if ( ! $this->ability_matches( $ability, $policy['abilities'] ?? [ '*' ] ) ) { continue; }
            if ( ! empty( $policy['agents'] ) && ! $this->agent_matches( $context['agent'] ?? '', $policy['agents'] ) ) { continue; }
            if ( ! empty( $policy['conditions'] ) && ! $this->conditions_match( $context, $policy['conditions'] ) ) { continue; }

            $elapsed = ( hrtime( true ) - $start ) / 1e6;
            $reason = $this->build_reason( $policy['reason'] ?? 'Matched policy.', $context );
            return new WPG_Policy_Result(
                $policy['decision'] ?? 'deny', $policy['id'] ?? 'unknown',
                policy_name: $policy['name'] ?? 'Unknown Policy', reason: $reason,
                elapsed_ms: $elapsed, context: $context, matched: $policy,
            );
        }

        $elapsed = ( hrtime( true ) - $start ) / 1e6;
        return new WPG_Policy_Result(
            decision: 'allow', policy_id: 'default', policy_name: 'Default Allow',
            reason: 'No matching policy found. Action allowed by default.',
            elapsed_ms: $elapsed, context: $context, matched: [],
        );
    }

    private function ability_matches( string $ability, array $patterns ): bool {
        foreach ( $patterns as $pattern ) {
            if ( '*' === $pattern ) { return true; }
            if ( str_ends_with( $pattern, '/*' ) ) {
                $ns = substr( $pattern, 0, -2 );
                if ( str_starts_with( $ability, $ns . '/' ) ) { return true; }
                continue;
            }
            if ( $ability === $pattern ) { return true; }
        }
        return false;
    }

    private function agent_matches( string $agent, array $patterns ): bool {
        foreach ( $patterns as $pattern ) {
            if ( '*' === $pattern ) { return true; }
            if ( strcasecmp( $agent, $pattern ) === 0 ) { return true; }
            if ( stripos( $agent, $pattern ) !== false ) { return true; }
        }
        return false;
    }

    private function conditions_match( array $context, array $conditions ): bool {
        foreach ( $conditions as $condition ) {
            if ( ! $this->check_condition( $context, $condition ) ) { return false; }
        }
        return true;
    }
    private function check_condition( array $context, array $condition ): bool {
        $field    = $condition['field'] ?? '';
        $operator = $condition['operator'] ?? 'equals';
        $expected = $condition['value'] ?? null;
        $actual   = $this->resolve_field( $context, $field );

        switch ( $operator ) {
            case 'equals': case 'eq': return $actual == $expected;
            case 'not_equals': case 'ne': return $actual != $expected;
            case 'greater_than': case 'gt': return is_numeric( $actual ) && $actual > $expected;
            case 'less_than': case 'lt': return is_numeric( $actual ) && $actual < $expected;
            case 'greater_than_or_equal': case 'gte': return is_numeric( $actual ) && $actual >= $expected;
            case 'less_than_or_equal': case 'lte': return is_numeric( $actual ) && $actual <= $expected;
            case 'contains': return is_string( $actual ) && str_contains( $actual, (string) $expected );
            case 'not_contains': return is_string( $actual ) && ! str_contains( $actual, (string) $expected );
            case 'in': return is_array( $expected ) && in_array( $actual, $expected, false );
            case 'not_in': return is_array( $expected ) && ! in_array( $actual, $expected, false );
            case 'exists': return $actual !== null;
            case 'not_exists': return $actual === null;
            case 'starts_with': return is_string( $actual ) && str_starts_with( $actual, (string) $expected );
            case 'ends_with': return is_string( $actual ) && str_ends_with( $actual, (string) $expected );
            default: return false;
        }
    }

    private function resolve_field( array $context, string $field ): mixed {
        $keys  = explode( '.', $field );
        $value = $context;
        foreach ( $keys as $key ) {
            if ( is_array( $value ) && array_key_exists( $key, $value ) ) {
                $value = $value[ $key ];
            } else { return null; }
        }
        return $value;
    }

    private function build_reason( string $template, array $context ): string {
        if ( ! str_contains( $template, '{' ) ) { return $template; }
        return preg_replace_callback( '/\{([a-zA-Z0-9_.]+)\}/', function ( array $m ) use ( $context ) {
            $v = $this->resolve_field( $context, $m[1] );
            if ( is_array( $v ) ) { return wp_json_encode( $v ); }
            return null !== $v ? (string) $v : $m[0];
        }, $template );
    }
    public function get_active_policies(): array {
        $this->ensure_loaded();
        return array_values( array_filter( $this->policies, fn( $p ) => ! empty( $p['active'] ) ) );
    }

    public function get_policy( string $id ): ?array {
        $this->ensure_loaded();
        foreach ( $this->policies as $p ) {
            if ( ( $p['id'] ?? '' ) === $id ) { return $p; }
        }
        return null;
    }

    public function save_policy( array $policy ): bool {
        $this->ensure_loaded();
        if ( empty( $policy['id'] ) ) {
            $policy['id'] = 'wpg-custom-' . sanitize_title( $policy['name'] ?? 'policy' ) . '-' . wp_rand( 1000, 9999 );
        }
        $found = false;
        foreach ( $this->policies as $i => $existing ) {
            if ( $existing['id'] === $policy['id'] ) {
                $this->policies[ $i ] = array_merge( $existing, $policy );
                $found = true;
                break;
            }
        }
        if ( ! $found ) { $this->policies[] = $policy; }
        $this->sort_policies();
        return update_option( 'wpg_policies', $this->policies );
    }

    public function delete_policy( string $id ): bool {
        $this->ensure_loaded();
        $count = count( $this->policies );
        $this->policies = array_values(
            array_filter( $this->policies, fn( $p ) => ( $p['id'] ?? '' ) !== $id )
        );
        if ( count( $this->policies ) === $count ) { return false; }
        return update_option( 'wpg_policies', $this->policies );
    }

    public function toggle_policy( string $id, bool $active ): bool {
        $this->ensure_loaded();
        foreach ( $this->policies as $i => $p ) {
            if ( ( $p['id'] ?? '' ) === $id ) {
                $this->policies[ $i ]['active'] = $active;
                return update_option( 'wpg_policies', $this->policies );
            }
        }
        return false;
    }

    public function install_default_policies(): void {
        $existing = get_option( 'wpg_policies', [] );
        if ( ! empty( $existing ) ) { return; }
        $dir = WPG_PLUGIN_DIR . 'policies/';
        if ( ! is_dir( $dir ) ) { return; }
        $policies = [];
        $json_files = glob( $dir . '*.json' );
        if ( false === $json_files ) { return; }
        foreach ( $json_files as $file ) {
            $content = file_get_contents( $file );
            if ( false === $content ) { continue; }
            $policy = json_decode( $content, true );
            if ( JSON_ERROR_NONE === json_last_error() && ! empty( $policy['id'] ) ) {
                $policies[] = $policy;
            }
        }
        usort( $policies, fn( $a, $b ) => ( $a['priority'] ?? 100 ) - ( $b['priority'] ?? 100 ) );
        update_option( 'wpg_policies', $policies );
    }

    private function ensure_loaded(): void {
        if ( $this->loaded ) { return; }
        $this->policies = get_option( 'wpg_policies', [] );
        if ( ! is_array( $this->policies ) ) { $this->policies = []; }
        $this->sort_policies();
        $this->loaded = true;
    }

    private function sort_policies(): void {
        usort( $this->policies, fn( $a, $b ) => ( $a['priority'] ?? 100 ) - ( $b['priority'] ?? 100 ) );
    }
}